/************************************************************************
	main.c

    Emma's Heart
    Copyright (C) 2011 Simon Inns

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

	Email: simon.inns@gmail.com

************************************************************************/

#ifndef MAIN_C
#define MAIN_C

// Global includes
#include <htc.h>

// Fuse configuration for the PIC12F683
__CONFIG(FCMEN_OFF & IESO_OFF & BOREN_ON & CP_OFF & MCLRE_ON &
	PWRTE_OFF & WDTE_OFF & FOSC_INTOSCIO);

// Define oscillator frequency
#define _XTAL_FREQ 8000000

// Hardware mapping definitions
#define BLUE	GP4
#define RED		GP2
#define GREEN	GP5

// Globals for PWM
unsigned char pwmCounter = 0;

int redActualBrightness = 0;
int redTargetBrightness = 0;
int redFadeCounter = 0;
int redFadeSpeed = 64;

int greenActualBrightness = 0;
int greenTargetBrightness = 0;
int greenFadeCounter = 0;
int greenFadeSpeed = 64;

int blueActualBrightness = 0;
int blueTargetBrightness = 0;
int blueFadeCounter = 0;
int blueFadeSpeed = 64;

// Interrupt Service Request handler
void interrupt isr()
{
	// Is this timer1 interrupting? (PWM sound generation interrupt)
	if (TMR1IF)
	{
		// RED ---------------------------------------------------
		// Perform the PWM brightness control
		if (redActualBrightness > pwmCounter)
			RED = 1; else RED = 0;
 
		// Perform fading control
		if (redTargetBrightness > redActualBrightness)
		{
			// Fading on
			redFadeCounter++;
			if (redFadeCounter == redFadeSpeed)
			{
				redActualBrightness++;
				redFadeCounter = 0;
			}	
		}	
		else
		{
			if (redTargetBrightness < redActualBrightness)
			{
				// Fading off
				redFadeCounter++;
				if (redFadeCounter == redFadeSpeed)
				{
					redActualBrightness--;
					redFadeCounter = 0;
				}
			}	
		}
		
		// GREEN -------------------------------------------------
		// Perform the PWM brightness control
		if (greenActualBrightness > pwmCounter)
			GREEN = 1; else GREEN = 0;
 
		// Perform fading control
		if (greenTargetBrightness > greenActualBrightness)
		{
			// Fading on
			greenFadeCounter++;
			if (greenFadeCounter == greenFadeSpeed)
			{
				greenActualBrightness++;
				greenFadeCounter = 0;
			}	
		}	
		else
		{
			if (greenTargetBrightness < greenActualBrightness)
			{
				// Fading off
				greenFadeCounter++;
				if (greenFadeCounter == greenFadeSpeed)
				{
					greenActualBrightness--;
					greenFadeCounter = 0;
				}
			}	
		}
		
		// BLUE --------------------------------------------------
		// Perform the PWM brightness control
		if (blueActualBrightness > pwmCounter)
			BLUE = 1; else BLUE = 0;
 
		// Perform fading control
		if (blueTargetBrightness > blueActualBrightness)
		{
			// Fading on
			blueFadeCounter++;
			if (blueFadeCounter == blueFadeSpeed)
			{
				blueActualBrightness++;
				blueFadeCounter = 0;
			}	
		}	
		else
		{
			if (blueTargetBrightness < blueActualBrightness)
			{
				// Fading off
				blueFadeCounter++;
				if (blueFadeCounter == blueFadeSpeed)
				{
					blueActualBrightness--;
					blueFadeCounter = 0;
				}
			}	
		}

		// Update the PWM counter
		pwmCounter++;
		if (pwmCounter > 31) pwmCounter = 0;
		
		// Reset the timer1 counter bytes
		TMR1H = 0xFD;
		TMR1L = 0x8E;

		// Reset the timer1 interrupt flag
		TMR1IF = 0;
	}
}

// Max, Half and Quarter intensities
#define MAX		31
#define HAF		16
#define QUA		8

// Steps in the sequence
unsigned char sequenceLength = 49;

// The sequence to be displayed (Red, Green, Blue, Number of Seconds)
const unsigned char colour[][4] = {
	MAX,   0,   0, 2,
	  2,   0,   0, 1,
	MAX,   0,   0, 2,
	  2,   0,   0, 1,	
	MAX,   0,   0, 2,
	  2,   0,   0, 1,
	MAX,   0,   0, 2,
	  2,   0,   0, 1,
	MAX,   0,   0, 2,
	  2,   0,   0, 1,
	MAX,   0,   0, 2,
	  2,   0,   0, 1,
	MAX,   0,   0, 2,
	  2,   0,   0, 1,	  
	
	MAX,   0,   0, 30,
	QUA, QUA, MAX, 30,
	HAF, MAX, MAX, 30,
	HAF, HAF, MAX, 30,
	MAX, QUA, HAF, 30,
	MAX, HAF,   0, 30,
	QUA,   0, MAX, 30,
	QUA, MAX,   0, 30,
	MAX,   0, MAX, 30,
	MAX, QUA, MAX, 30,
	MAX, HAF, HAF, 30,
	  0, MAX,   0, 30,
	  0, MAX, MAX, 30,
	MAX,   0, HAF, 30,
	MAX, MAX, QUA, 30,
	  0,   0, MAX, 30,
	HAF, QUA, MAX, 30,
	HAF, MAX, HAF, 30,
	MAX, HAF, MAX, 30,
	  0, HAF, MAX, 30,
	QUA, MAX, MAX, 30,
	MAX, QUA, QUA, 30,
	QUA, MAX, QUA, 30,
	HAF,   0, MAX, 30,
	  0, MAX, HAF, 30,
	MAX, MAX, HAF, 30,
	QUA, MAX, HAF, 30,
	HAF, MAX,   0, 30,
	  0, MAX, QUA, 30,
	MAX, HAF, QUA, 30,
	MAX, QUA,   0, 30,
	MAX,   0, QUA, 30,
	HAF, MAX, QUA, 30,
	  0, QUA, MAX, 30,
	QUA, HAF, MAX, 30,
	MAX, MAX,   0, 30
};

// Pre-calculated white balance (in RGB triplets)
// This chart is calculated based on the relative brightness of the 
// 3 LEDs (based on the dimmest LED which is blue)
const unsigned char whiteBalance[][3] = 
{
	0,0,0,
	1,1,1,
	2,2,2,
	3,3,3,
	4,3,4,
	5,4,5,
	6,5,6,
	7,5,7,
	7,6,8,
	8,7,9,
	9,7,10,
	10,8,11,
	11,9,12,
	12,9,13,
	13,10,14,
	14,11,15,
	15,11,16,
	15,12,17,
	16,13,18,
	17,14,19,
	18,14,20,
	19,15,21,
	20,16,22,
	21,16,23,
	22,17,24,
	22,18,25,
	23,18,26,
	24,19,27,
	25,20,28,
	26,20,29,
	27,21,30,
	28,22,31
};

// Main procedure
void main(void)
{
	OSCCON = 0b01110001;	// 8Mhz INT OSC
	
	// Set up the PIC12F683 IO pins
	GPIO   = 0b00000000;	// Set all pins to zero
	ANSEL  = 0b00000000;	// Disable analogue inputs
	TRISIO = 0b00000000;	// Set all IO pins to output
	OPTION_REG = 0b00000000;	// Set the option register (page 12 of DS)
	WPU    = 0b00000000;	// Set weak pull-up off on all pins
	CMCON0 = 7;				// Disable comparator 0

	// Set up the PWM control interrupt on timer1

	// Set up the timer
	TMR1IF = 0; // Reset the timer1 interrupt flag
	nT1SYNC = 0;
	T1CON = 0b00000001;
	TMR1IE = 1; // Turn on the timer1 interrupt

	// We want 50Hz PWM with 32 brightness levels:
	// 1,000,000 uS / 100Hz = 10,000
	// 10,000 / 32 steps = 312.5 uS
	//
	// Fosc/4 is 2,000,000 meaning there are 2 timer ticks
	// per uS.  We want an interrupt every 312.5 uS so
	// every 625 clock ticks.
	//
	// With a prescale of 1:1 this means:
	// 65535 - 625 = 64910 or 0xFD8E
	TMR1H = 0xFD;
	TMR1L = 0x8E;
	
	// Enable the interrupts
	PEIE = 1;
	GIE = 1;

	// Main loop
	while(1)
	{
		// Cycle throught the available colour mixes
		for (unsigned char step = 0; step < sequenceLength; step++)
		{
			// Select the LED brightness based on the white balance table:
			redTargetBrightness = whiteBalance[colour[step][0]][0];
			greenTargetBrightness = whiteBalance[colour[step][1]][1];
			blueTargetBrightness = whiteBalance[colour[step][2]][2];
	
			// Pause for the specified number of seconds
			for (unsigned int seconds = 0; seconds < colour[step][3]; seconds++)
				for (unsigned int delay = 0; delay < 10000; delay++) __delay_us(100);
		}
	}		
}

#endif